/*
 * Kylin-video
 *
 * Copyright (C) 2021, Tianjin KYLIN Information Technology Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Liu Cong <liucong1@kylinos.cn>
 *
 */

#include "eventpasswidget.h"

#include <QTimer>
#include <QDebug>
#include <QTextBlock>
#include "global/globalsignal.h"
#include "global/global.h"

using namespace Global;

EventPassWidget::EventPassWidget(QWidget *parent) : QWidget(parent)
{
    setAttribute(Qt::WA_TranslucentBackground);
    setMouseTracking(true);

    m_mouseUsed = true;

    initGlobalSig();

    m_checkMouseTimer = new QTimer;
    m_checkMouseTimer->setInterval(2000);
    connect(m_checkMouseTimer, &QTimer::timeout, [&](){
        if(!m_hasVideo)
            return;
        if(m_isMouseEnter) {
            if (m_mouseUsed)
                setCursor(QCursor(Qt::BlankCursor));
        }
        g_user_signal->hideBar(true);

        m_checkMouseTimer->stop();
        return;
    });

    m_lMouseClickTimer = new QTimer;
    m_lMouseClickTimer->setInterval(300);
    connect(m_lMouseClickTimer, &QTimer::timeout, [&](){
        emit mousePressed();
        m_lMouseClickTimer->stop();
    });
}

void EventPassWidget::setMouseUsed(bool used)
{
    m_mouseUsed = used;
    if (!m_mouseUsed) {
        setCursor(QCursor(Qt::ArrowCursor));
        m_checkMouseTimer->stop();
    }
}

void EventPassWidget::playStateChange(Mpv::PlayState state)
{
    m_playState = state;
    if(m_playState < 0) {
        m_checkMouseTimer->stop();
    }
    else {
        m_checkMouseTimer->start();
    }
}

void EventPassWidget::videIdChange(int vid)
{
    if(vid < 0)
        m_hasVideo = false;
    else
        m_hasVideo = true;
}

void EventPassWidget::initGlobalSig()
{
    connect(g_core_signal, &GlobalCoreSignal::sigStateChange, this, &EventPassWidget::playStateChange);
    connect(g_core_signal, &GlobalCoreSignal::sigVideoIdChange, this, &EventPassWidget::videIdChange);
}

void EventPassWidget::mousePressEvent(QMouseEvent *e)
{
    if (e->buttons() == Qt::RightButton) {
        m_isMouseEnter = false;
        g_user_signal->showRightMenu();
    }
    else if (e->buttons() == Qt::LeftButton) {
        if (!m_mouseUsed)
            return;
        m_lMouseClickTimer->start();
    }
}

void EventPassWidget::mouseMoveEvent(QMouseEvent *e)
{
    m_isMouseEnter = true;
    m_checkMouseTimer->stop();
    if (cursor().shape() != Qt::ArrowCursor) {
        setCursor(QCursor(Qt::ArrowCursor));
    }
    g_user_signal->hideBar(false);
    if(m_playState > 0)
        m_checkMouseTimer->start();
}

void EventPassWidget::mouseDoubleClickEvent(QMouseEvent *e)
{
    if (!m_mouseUsed)
        return;
    if(e->buttons() == Qt::LeftButton) {
        g_user_signal->fullScreen();
        m_lMouseClickTimer->stop();
    }
}

void EventPassWidget::enterEvent(QEvent *e)
{
    m_isMouseEnter = true;
}

void EventPassWidget::leaveEvent(QEvent *e)
{
    m_isMouseEnter = false;
    if(m_playState > 0)
        m_checkMouseTimer->start();
}
