//***************************************************************************
//
//  Copyright (c) 2002 - 2006 Intel Corporation
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//***************************************************************************

/**
	@file	IFXMeshMap.h
*/

#ifndef IFXMeshMap_H
#define IFXMeshMap_H

#include "IFXDataTypes.h"
#include "IFXResult.h"
#include "IFXAuthorMesh.h"
#include "IFXAuthorLineSet.h"
#include "IFXAuthorPointSet.h"

class IFXAuthorMeshMap;
class IFXVertexMap;

// {DF99C4A8-FEE3-4351-86BF-22C7AA93EDBB}
IFXDEFINE_GUID(IID_IFXMeshMap,
0xdf99c4a8, 0xfee3, 0x4351, 0x86, 0xbf, 0x22, 0xc7, 0xaa, 0x93, 0xed, 0xbb);

/** 
	This class describes the mapping from an AuthorMesh to an IFXMeshGroup.
*/
class IFXMeshMap : public IFXUnknown
{
public:

	/** 
		Allocates a map of sufficient size to fully describe the mapping of pMesh.
	*/
	virtual IFXRESULT IFXAPI Allocate(IFXAuthorMesh* pMesh) = 0;

	/** 
		Allocates a map of sufficient size to fully describe the mapping of pLineSet.
	*/
	virtual IFXRESULT IFXAPI Allocate(IFXAuthorLineSet* pLineSet) = 0;

	/** 
		Allocates a map of sufficient size to fully describe the mapping of pPointSet.
	*/
	virtual IFXRESULT IFXAPI Allocate(IFXAuthorPointSet* pPointSet) = 0;

	/** 
		Allocates a map of sufficient size to fully describe the mapping of pMesh. 
	*/
	virtual IFXRESULT IFXAPI Allocate(IFXMeshMap* pMeshMap) = 0;

	/** 
		Combines this map with an AuthorMeshMap to provide a complete mapping from 
		an orginal author mesh to the final IFXMesh.
		This map is modified in place.

		@note each map in pMeshMap must be less than or equal to corresponding map
	*/
	virtual IFXRESULT IFXAPI Concatenate(IFXAuthorMeshMap* pAuthorMeshMapB) = 0;

	/** 
		Combines this map with another IFXMeshMap and puts the result in outMap.
		The input mesh map should be the one from earlier in the stream,
		probably the one generated by the compressor.  "this" should be the map
		generated by the IFXMeshCompiler.
	*/
	virtual IFXRESULT IFXAPI ConcatenateMeshMap(
									IFXMeshMap* pMeshMap,
									IFXMeshMap** ppOutMap) = 0;

	/** 
		Adds the contents of an IFXAuthorMeshMap to an IFXMeshMap.  This
		is really just intended as a way to convert an IFXAuthorMeshMap
		to an IFXMeshMap (load a blank IFXMeshMap from an IFXAuthorMeshMap)
		for use in generating a master mapping table. 
	*/
	virtual	IFXRESULT IFXAPI PopulateMeshMap(IFXAuthorMeshMap* pAuthorMeshMap) = 0;

	/** 
		This allows an identity map to be added to an IFXVertexMap that will
		not be used for another purpose (useful in the case that only a
		few of the maps in the IFXMeshMap are used by the person who allocates it).
	*/
	virtual IFXRESULT IFXAPI AddIdentityMappingToMap(U32 mapIndex) = 0;

	/** 
		Allows to modify maps arrays. Adds mapping to an IFXVertexMap.

		@param	mapIndex		Iindex of map array.
		@param	origVertexIndex	Index of original vertex.
		@param	indexMesh		Mesh index.
		@param	indexVertex		Vertex index.
	*/
	virtual IFXRESULT IFXAPI AddMappingToMap(
								const U32 mapIndex, const U32 origVertexIndex, 
								const U32 indexMesh, U32 indexVertex) = 0;

	/**
		Returns pointer to array of U32, modify this array to describe mapping
	
		@note	If you want to update maps do not use this method. 
				Use AddMappingToMap instead.
	*/
	virtual IFXVertexMap* IFXAPI GetFaceMap() = 0;
	virtual IFXVertexMap* IFXAPI GetPositionMap() = 0;
	virtual IFXVertexMap* IFXAPI GetNormalMap() = 0;
	virtual IFXVertexMap* IFXAPI GetTextureMap() = 0;
	virtual IFXVertexMap* IFXAPI GetDiffuseMap() = 0;
	virtual IFXVertexMap* IFXAPI GetSpecularMap() = 0;
	
	/**
		@note	If you want to update maps do not use this method. 
				Use AddMappingToMap instead.

			0 - face map
			1 - position map
			2 - normal map
			3 - texture map
			4 - diffuse map
			5 - specular map
	*/
	virtual IFXVertexMap* IFXAPI GetMap(U32 i) = 0;
};

#endif
