# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from ConfigParser import ConfigParser, NoSectionError
from gtk.gdk import CONTROL_MASK
from gtk import (STOCK_DIRECTORY, settings_get_default, AccelGroup,
                 accel_map_change_entry)
from random import randrange
from os.path import join, isdir
from os import makedirs

from common.config import ConfigLoader

from modules.explorer.views import Views
from modules.explorer.config import Config
from modules.explorer.musicdb import MusicDatabase

class Explorer:
    def __init__(self, extensionsloader):
        self.extensions = extensionsloader
        self.module = {'name': 'Explorer',
                       'logo': STOCK_DIRECTORY,
                       'configurable': True}

        # Start configuration
        name = self.module['name'].lower() + '.cfg'
        self.config = ConfigLoader()
        self.config_file = join(self.config.confdir, 'modules', name)
        self.data_dir = join(self.config.datadir, 'modules', 'explorer')

        if not isdir(self.data_dir):
            makedirs(self.data_dir)

    def start_module(self):
        """This function starts the module."""
        def launch_explorer(glade_file):
            """This function launch the explorer."""
            self.widgets = glade_file

            # Create the View object
            config = self.get_configuration()
            self.view = Views(glade_file, config)

            self.cfg = Config(self.module, self.view, self.config_file,
                                                      self.get_configuration)

            pbar = glade_file.get_widget('progressbar1')
            self.musicdb = MusicDatabase(config, pbar)

            # Change view menu
            menubar = self.widgets.get_widget('viewmenu')
            views = dict()

            settings = settings_get_default()
            settings.set_property('gtk-can-change-accels', True)
            ag = AccelGroup()
            menubar.set_accel_group(ag)
            window = self.widgets.get_widget('window1')
            window.add_accel_group(ag)

            # Add accelerators
            menubar.set_accel_path('<main>/_Show')

            accel_map_change_entry('<main>/_Show/Lightweight',
                                   ord('l'), CONTROL_MASK, True)
            accel_map_change_entry('<main>/_Show/Basic',
                                   ord('b'), CONTROL_MASK, True)
            accel_map_change_entry('<main>/_Show/Normal',
                                   ord('n'), CONTROL_MASK, True)
            accel_map_change_entry('<main>/_Show/Full',
                                   ord('f'), CONTROL_MASK, True)
            accel_map_change_entry('<main>/_Show/Albums',
                                   ord('u'), CONTROL_MASK, True)
            accel_map_change_entry('<main>/_Show/Playlists',
                                   ord('p'), CONTROL_MASK, True)
            accel_map_change_entry('<main>/_Show/Webradios',
                                   ord('w'), CONTROL_MASK, True)

            # Connect menu items
            views['lightweight'] = self.widgets.get_widget('menu-view-lightweight')
            views['basic'] = self.widgets.get_widget('menu-view-basic')
            views['normal'] = self.widgets.get_widget('menu-view-normal')
            views['full'] = self.widgets.get_widget('menu-view-full')
            views['albums'] = self.widgets.get_widget('menu-view-albums')
            views['playlists'] = self.widgets.get_widget('menu-view-playlists')
            views['webradios'] = self.widgets.get_widget('menu-view-webradio')

            # Check the radio button
            if config['mode'] == 'lightweight':
                views['lightweight'].set_active(True)
            elif config['mode'] == 'basic':
                views['basic'].set_active(True)
            elif config['mode'] == 'normal':
                views['normal'].set_active(True)
            elif config['mode'] == 'full':
                views['full'].set_active(True)
            elif config['mode'] == 'albums':
                views['albums'].set_active(True)
            elif config['mode'] == 'playlists':
                views['playlists'].set_active(True)
            else:
                views['webradios'].set_active(True)

            def on_view_change(widget, view_name):
                conf = self.get_configuration()
                Views(self.widgets, conf, view_name)

            for view in views:
                views[view].connect('activate', on_view_change, view)

        def play_item():
            songs_list = self.musicdb.load()
            if len(songs_list) > 1:
                index = randrange(len(songs_list) - 1)
                song = songs_list[index]

                self.extensions.load_event('OnPlayNewSong', song)

        def configuration(args):
            """Load the configuration interface."""
            self.cfg.configuration(args)

        def save_config(args):
            """Load the configuration interface."""
            self.cfg.save_config(args)

        # Connect to Bluemindo's signals
        self.extensions.connect('OnBluemindoStarted', launch_explorer)
        self.extensions.connect('OnPlayPressedWithoutQueue', play_item)
        self.extensions.connect('OnModuleConfiguration', configuration)
        self.extensions.connect('OnModuleConfigurationSave', save_config)

    def get_configuration(self):
        configparser = ConfigParser()
        configparser.read(self.config_file)
        config = {}

        try:
            for item in configparser.items(self.module['name']):
                if item[1].isdigit():
                    value = int(item[1])
                else:
                    value = str(item[1])

                config[item[0]] = value
        except NoSectionError:
            config['repeat'] = True
            config['artwork'] = True
            config['scan'] = False
            config['mode'] = 'full'
            config['folder'] = ''
            config['columns'] = 'track,title,artist,album,length'

        name = self.module['name'].lower() + '.cfg'
        config['__config-file'] = self.config_file
        config['__data-dir'] = self.data_dir
        config['__module-name'] = self.module['name']
        config['__extensions'] = self.extensions

        return config