% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chebyshev.R
\name{chebyshev}
\alias{chebyshev}
\alias{chebyshev_roots}
\title{Chebyshev polynomials}
\usage{
chebyshev(degree, kind = "t", indeterminate = "x", normalized = FALSE)

chebyshev_roots(k, n)
}
\arguments{
\item{degree}{degree of polynomial}

\item{kind}{\code{"t"} or \code{"u"} (Chebyshev polynomials of the first and
second kinds), or \code{"c"} or \code{"s"}}

\item{indeterminate}{indeterminate}

\item{normalized}{provide normalized coefficients}

\item{k, n}{the k'th root of the n'th chebyshev polynomial}
}
\value{
a mpoly object or mpolyList object
}
\description{
Chebyshev polynomials as computed by orthopolynom.
}
\examples{

chebyshev(0)
chebyshev(1)
chebyshev(2)
chebyshev(3)
chebyshev(4)
chebyshev(5)
chebyshev(6)
chebyshev(10)

chebyshev(0:5)
chebyshev(0:5, normalized = TRUE)
chebyshev(0:5, kind = "u")
chebyshev(0:5, kind = "c")
chebyshev(0:5, kind = "s")
chebyshev(0:5, indeterminate = "t")



# visualize the chebyshev polynomials

library(ggplot2); theme_set(theme_classic())
library(tidyr)

s <- seq(-1, 1, length.out = 201)
N <- 5 # number of chebyshev polynomials to plot
(cheb_polys <- chebyshev(0:N))

# see ?bernstein for a better understanding of
# how the code below works

df <- data.frame(s, as.function(cheb_polys)(s))
names(df) <- c("x", paste0("T_", 0:N))
mdf <- gather(df, degree, value, -x)
qplot(x, value, data = mdf, geom = "line", color = degree)



# roots of chebyshev polynomials
N <- 5
cheb_roots <- chebyshev_roots(1:N, N)
cheb_fun <- as.function(chebyshev(N))
cheb_fun(cheb_roots)



# chebyshev polynomials are orthogonal in two ways:
T2 <- as.function(chebyshev(2))
T3 <- as.function(chebyshev(3))
T4 <- as.function(chebyshev(4))

w <- function(x) 1 / sqrt(1 - x^2)
integrate(function(x) T2(x) * T3(x) * w(x), lower = -1, upper = 1)
integrate(function(x) T2(x) * T4(x) * w(x), lower = -1, upper = 1)
integrate(function(x) T3(x) * T4(x) * w(x), lower = -1, upper = 1)

(cheb_roots <- chebyshev_roots(1:4, 4))
sum(T2(cheb_roots) * T3(cheb_roots) * w(cheb_roots))
sum(T2(cheb_roots) * T4(cheb_roots) * w(cheb_roots))
sum(T3(cheb_roots) * T4(cheb_roots) * w(cheb_roots))

sum(T2(cheb_roots) * T3(cheb_roots))
sum(T2(cheb_roots) * T4(cheb_roots))
sum(T3(cheb_roots) * T4(cheb_roots))


}
\seealso{
\code{\link[orthopolynom:chebyshev.t.polynomials]{orthopolynom::chebyshev.t.polynomials()}},
\code{\link[orthopolynom:chebyshev.u.polynomials]{orthopolynom::chebyshev.u.polynomials()}},
\code{\link[orthopolynom:chebyshev.c.polynomials]{orthopolynom::chebyshev.c.polynomials()}},
\code{\link[orthopolynom:chebyshev.s.polynomials]{orthopolynom::chebyshev.s.polynomials()}},
\url{https://en.wikipedia.org/wiki/Chebyshev_polynomials}
}
\author{
David Kahle calling code from the orthopolynom package
}
