/*-----------------------------------------------------------------------

                         SYRTHES version 3.4
                         -------------------

     This file is part of the SYRTHES Kernel, element of the
     thermal code SYRTHES.

     Copyright (C) 1988-2009 EDF S.A., France

     contact: syrthes-support@edf.fr


     The SYRTHES Kernel is free software; you can redistribute it
     and/or modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2 of
     the License, or (at your option) any later version.

     The SYRTHES Kernel is distributed in the hope that it will be
     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.


     You should have received a copy of the GNU General Public License
     along with the Code_Saturne Kernel; if not, write to the
     Free Software Foundation, Inc.,
     51 Franklin St, Fifth Floor,
     Boston, MA  02110-1301  USA

-----------------------------------------------------------------------*/
/* Conversions de maillages en format Syrthes */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "convert2syrthes.h"


extern char *basename (__const char *__filename) __THROW __nonnull ((1));

static int cmpentier(const void *p1, const void *p2){
/* fonction servant pour l'utilisation de qsort...*/
 return (* (int const *) p1 - * (int const *) p2);
}

int calcul_reffaces(struct typ_maillage *maillage) {
/* Pour Syrthes 3.4 : calcul des references des faces */
/* a partir de liste_elembord et liste_elem */
/* - classer dans un tableau local par ordre croissant les 4 premiers noeuds si dim_elem=3, les 3 premiers si dim_elem=2 */
/* - Pour chaque element de bord, chercher l'element auquel il appartient et en deduire le numero de face */

    int i,j,k; /* indices de boucles */
    int nbfaces_par_elem; /* nombre de faces par element : 4 si dim_elem=3, 3 si dim_elem=2 */
    int **elem_noeudstries; /* tableau des elements ou pour chacun les references aux noeuds sont tires par ordre croissant */
    int nbnoeuds_sommets; /* nombre de noeuds sommets par element */
    int min, posmin, max, posmax, posmilieu; /* pour calculer min max */
    int ktrouve; /* element trouve */
    int numnoeud_nonpris; /* noeud de l'element ne faisant pas partie de l'element de bord qui appartient a cet element */
    int numero;

    printf("  Face references calculation...\n");
    
    /* nombre de faces par element et nombre de noeuds sommets par element */
    if ((*maillage).dim_elem == 3) nbfaces_par_elem = 4;
    else nbfaces_par_elem = 3;
    /* nombre de noeuds sommets par element */
    nbnoeuds_sommets = nbfaces_par_elem;

    /* allocation du tableau des references des faces */
    (*maillage).liste_reffaces = (int **) malloc (sizeof(int *) * ((*maillage).nbelem));
    if ((*maillage).liste_reffaces == NULL) {
      printf("ERROR : allocation error for face references, needed size : %i \n", sizeof(int *) * ((*maillage).nbelem));
      return 1;
    }
    for  (i=0;i<(*maillage).nbelem;i++) (*maillage).liste_reffaces[i] = (int *) malloc (sizeof(int) * nbfaces_par_elem);
    /* verification de la derniere allocation */
    if ((*maillage).liste_reffaces[(*maillage).nbelem-1] == NULL) {
      printf("ERROR : allocation error for face references (last allocation)\n");
      return 1;
    }
    /* initialisation des references aux faces */
    for  (i=0;i<(*maillage).nbelem;i++) {
      for  (j=0;j<nbfaces_par_elem;j++) {
        (*maillage).liste_reffaces[i][j] = 0;
      }
    }
    
    elem_noeudstries = (int **) malloc (sizeof(int *) * ((*maillage).nbelem));
    if (elem_noeudstries == NULL) {
      printf("ERROR : allocation error for elem_noeudstries, needed size : %i \n", sizeof(int *) * ((*maillage).nbelem));
      return 1;
    }
    
    /* tri par ordre croissant */
    for (i=0;i<(*maillage).nbelem;i++) {
      /* allocation tableau des elemnts avec noeuds tries */
      elem_noeudstries[i] = (int *) malloc (sizeof(int ) * nbnoeuds_sommets);
      /* initialisation */
      for (j=0;j<nbnoeuds_sommets;j++) elem_noeudstries[i][j]=(*maillage).liste_elem[i][j];
      /* tri */
      qsort(&elem_noeudstries[i][0],nbnoeuds_sommets,sizeof(int),cmpentier);
    }
    
    /* pour tous les elements de bord, recherche de la face */
    if (nbnoeuds_sommets == 4) { /* cas tetraedres */
      for (i=0;i<(*maillage).nbelem_de_bord;i++) {
        /* soit 2 soit 3 noeuds, classement par min */
        min = (*maillage).liste_elembord[i][0];
        posmin = 0;
        max = (*maillage).liste_elembord[i][nbnoeuds_sommets-2];
        posmax = nbnoeuds_sommets-2;
        for (j=0;j<nbnoeuds_sommets-1;j++) {
          if ((*maillage).liste_elembord[i][j] > max) {
  	    max = (*maillage).liste_elembord[i][j] ;
	    posmax=j;
	  }
          if ((*maillage).liste_elembord[i][j] < min) {
	    min = (*maillage).liste_elembord[i][j] ;
	    posmin=j;
	  }	
        }
        switch (posmin+posmax) {
        case 1 : posmilieu = 2; break;
        case 2 : posmilieu = 1; break;
        case 3 : posmilieu = 0; break;
        default : printf("ERROR : for posmilieu = %i during face references calculation\n",posmilieu); return 1;
        }
        ktrouve = -1;
        /* posmin est la position du plus petit noeud dans la liste liste_elembord[i][] */
        /* posmax est la position du plus grand noeud dans la liste liste_elembord[i][] */
        /* posmilieu est la position du troisieme noeud dans la liste liste_elembord[i][] */
        for (k=0;k<(*maillage).nbelem;k++) {
          if (elem_noeudstries[k][0] == (*maillage).liste_elembord[i][posmin]) {
	    /* cet element contient le plus petit noeud de l'element de bord i */
            if (elem_noeudstries[k][1] == (*maillage).liste_elembord[i][posmilieu]) {
	      if (elem_noeudstries[k][2] == (*maillage).liste_elembord[i][posmax]) {
	        /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1 N2 N3 N4nonpris */
	        ktrouve = k;
		numnoeud_nonpris = elem_noeudstries[k][3];
		exit;
	      }
	      else if (elem_noeudstries[k][3] == (*maillage).liste_elembord[i][posmax]) {
	        /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1 N2 N3nonpris N4 */
	        ktrouve = k;
		numnoeud_nonpris = elem_noeudstries[k][2];
		exit;
	      }
	    }
	    else if (( elem_noeudstries[k][2] == (*maillage).liste_elembord[i][posmilieu]) && 
	             ( elem_noeudstries[k][3] == (*maillage).liste_elembord[i][posmax])) {
	      /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1 N2nonpris N3 N4 */
	      ktrouve = k;
	      numnoeud_nonpris = elem_noeudstries[k][1];
	      exit;
	    }
	  }
	  else if ((elem_noeudstries[k][1] == (*maillage).liste_elembord[i][posmin]) &&
	           (elem_noeudstries[k][2] == (*maillage).liste_elembord[i][posmilieu]) &&
		   (elem_noeudstries[k][3] == (*maillage).liste_elembord[i][posmax])){
	    /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1nonpris N2 N3 N4 */
	    ktrouve = k;
	    numnoeud_nonpris = elem_noeudstries[k][0];
	    exit;
	  }
        }
	/* deduction de la face selon format Syrthes : affectation de la couleur de l'element de bord*/
	for (j=0;j<nbnoeuds_sommets;j++) {
	  if ((*maillage).liste_elem[ktrouve][j] == numnoeud_nonpris) {
	    /* numero de noeud non present dans l'element de bord */
	    numero = j+1;
	    exit;
	  }
	}
	/* numero 1 -> face 4 */
	/* numero 2 -> face 3 */
	/* numero 3 -> face 2 */
	/* numero 4 -> face 1 */
	(*maillage).liste_reffaces[ktrouve][nbnoeuds_sommets-numero] = (*maillage).coul_elembord[i];
      }
    }
    else if (nbnoeuds_sommets == 3) {/* cas triangles : posmilieu n'existe pas*/
      for (i=0;i<(*maillage).nbelem_de_bord;i++) {
        /* soit 2 soit 3 noeuds, classement par min */
	if ((*maillage).liste_elembord[i][0] > (*maillage).liste_elembord[i][1]) {
	  posmax = 0;
	  posmin = 1;
	}
	else {
	  posmax = 1;
	  posmin = 0;
	}
        ktrouve = -1;
        /* posmin est la position du plus petit noeud dans la liste liste_elembord[i][] */
        /* posmax est la position du plus grand noeud dans la liste liste_elembord[i][] */
        for (k=0;k<(*maillage).nbelem;k++) {
          if (elem_noeudstries[k][0] == (*maillage).liste_elembord[i][posmin]) {
	    /* cet element contient le plus petit noeud de l'element de bord i */
            if (elem_noeudstries[k][1] == (*maillage).liste_elembord[i][posmax]) {
	      /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1 N2 N3nonpris */
	      ktrouve = k;
	      numnoeud_nonpris = elem_noeudstries[k][2];
	      exit;
	    }
	    else if (elem_noeudstries[k][2] == (*maillage).liste_elembord[i][posmax]) {
	      /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1 N2nonpris N3 */
	      ktrouve = k;
	      numnoeud_nonpris = elem_noeudstries[k][1];
	      exit;
	    }
	  }
	  else if ((elem_noeudstries[k][1] == (*maillage).liste_elembord[i][posmin]) &&
		   (elem_noeudstries[k][2] == (*maillage).liste_elembord[i][posmax])){
	    /* on a trouve l'element qui correspond a l'element de bord : noeuds tries N1nonpris N2 N3 */
	    ktrouve = k;
	    numnoeud_nonpris = elem_noeudstries[k][0];
	    exit;
	  }
        }
	
	/* deduction de l'arete selon format Syrthes : affectation de la couleur de l'element de bord*/
	for (j=0;j<nbnoeuds_sommets;j++) {
	  if ((*maillage).liste_elem[ktrouve][j] == numnoeud_nonpris) {
	    /* numero de noeud non present dans l'element de bord */
	    numero = j+1;
	    exit;
	  }
	}
	/* numero 1 -> arete 2 */
	/* numero 2 -> arete 3 */
	/* numero 3 -> arete 1 */
	(*maillage).liste_reffaces[ktrouve][numero%3] = (*maillage).coul_elembord[i];
      }
    }
    
    /* liberation de la memoire cas Syrthes 3.4 */
    for (i=0;i<(*maillage).nbelem;i++) free(elem_noeudstries[i]);
    free(elem_noeudstries);

    return 0;
  
}


int conversion_neu(struct typ_maillage *maillage, char* nomfich) {
/* lecture format Gambit Neutral, conversion au format Syrthes, ecriture au format Syrthes */
   if (ouvrir_neu(maillage, nomfich) != 0) return 1;
   if (lire_entete_neu(maillage) != 0) return 1;
   if (ecrire_entete_syr(maillage) != 0) return 1;
   if (lire_coord_neu(maillage) != 0) return 1;
   if (lire_noeuds_colores_neu(maillage) != 0) return 1;
   if (ecrire_coord_syr(maillage) != 0) return 1;
   if (lire_elem_neu(maillage) != 0) return 1;
   if (lire_zones_neu(maillage) != 0) return 1;
   if (ecrire_elem_syr(maillage) != 0) return 1;
   if (lire_cl_neu(maillage) != 0) return 1;
   if ((*maillage).version_syr == 4) {  /* version Syrthes 4.0 */
     if ( ((*maillage).dimension == 3 && (*maillage).dim_elem == 3)
       || ((*maillage).dimension == 2 && (*maillage).dim_elem == 2) ) {
	if(ecrire_elem_de_bord_syr(maillage) != 0) return 1;
     }
   }
   else {                               /* version Syrthes 3.4 */
     if ( ((*maillage).dimension == 3 && (*maillage).dim_elem == 3)
            || ((*maillage).dimension == 2 && (*maillage).dim_elem == 2) ) {
	if(ecrire_reffaces_syr(maillage) != 0) return 1;
     }
   }
   if (fermer_neu(maillage) != 0) return 1;
   return 0;
}

int conversion_msh(struct typ_maillage *maillage, char* nomfich) {
/* lecture format GMSH, conversion au format Syrthes, ecriture au format Syrthes */
   if (ouvrir_msh(maillage, nomfich) != 0) return 1;
   if (lire_entete_msh(maillage) != 0) return 1;
   if (ecrire_entete_syr(maillage) != 0) return 1;
   if (lire_coord_msh(maillage) != 0) return 1;
   if (lire_noeuds_colores_msh(maillage) != 0) return 1;
   if (ecrire_coord_syr(maillage) != 0) return 1;
   if (lire_elem_msh(maillage) != 0) return 1;
   if (lire_zones_msh(maillage) != 0) return 1;
   if (ecrire_elem_syr(maillage) != 0) return 1;
   if (lire_cl_msh(maillage) != 0) return 1;
   if ((*maillage).version_syr == 4) {  /* version Syrthes 4.0 */
     if ( ((*maillage).dimension == 3 && (*maillage).dim_elem == 3)
       || ((*maillage).dimension == 2 && (*maillage).dim_elem == 2) ) {
	if(ecrire_elem_de_bord_syr(maillage) != 0) return 1;
     }
   }
   else {                               /* version Syrthes 3.4 */
     if ( ((*maillage).dimension == 3 && (*maillage).dim_elem == 3)
       || ((*maillage).dimension == 2 && (*maillage).dim_elem == 2) ) {
	if(ecrire_reffaces_syr(maillage) != 0) return 1;
     }
   }
   if (fermer_msh(maillage) != 0) return 1;
   return 0;
}

#ifdef MED
int conversion_med(struct typ_maillage *maillage, char* nomfich, char* nomsyr) {
/* lecture format MED, conversion au format Syrthes, ecriture au format Syrthes */
  if (ouvrir_med(maillage, nomfich, nomsyr) != 0) return 1;
   if (lire_entete_med(maillage) != 0) return 1;
   if (ecrire_entete_syr(maillage) != 0) return 1;
   if (lire_coord_med(maillage) != 0) return 1;
   if (lire_noeuds_colores_med(maillage) != 0) return 1;
   if (ecrire_coord_syr(maillage) != 0) return 1;
   if (lire_elem_med(maillage) != 0) return 1;
   if (lire_zones_med(maillage) != 0) return 1;
   if (ecrire_elem_syr(maillage) != 0) return 1;
   if (lire_cl_med(maillage) != 0) return 1;
   if ((*maillage).version_syr == 4) {  /* version Syrthes 4.0 */
     if ( ((*maillage).dimension == 3 && (*maillage).dim_elem == 3)
       || ((*maillage).dimension == 2 && (*maillage).dim_elem == 2) ) {
	if(ecrire_elem_de_bord_syr(maillage) != 0) return 1;
     }
   }
   else {                               /* version Syrthes 3.4 */
     if ( ((*maillage).dimension == 3 && (*maillage).dim_elem == 3)
            || ((*maillage).dimension == 2 && (*maillage).dim_elem == 2) ) {
	if(ecrire_reffaces_syr(maillage) != 0) return 1;
     }
   }
   if (fermer_med(maillage) != 0) return 1;
   return 0;
}
#endif

int main(int iargc, char **iargv) {

  struct typ_maillage maillage;     /* structure de maillage */
  int extension; /* extension : 1 neu, 2 msh, 3 med */
  int m; /* position de l'option -m */
  int v; /* position de l'option -v */
  int o; /* position de l'option -o */
  int ref; /* position de l'option -ref */
  int dim; /* position de l'option -dim dimension si GMSH */
  int longueur; /* longueur de chaine de caracteres */
  char *nom_syr; /* base du nom de fichier d'entree */
  char *buffer;

  printf("\n***************************************************************\n");
  printf("  convert2syrthes : convert mesh file to SYRTHES mesh format     \n");
  printf("**************************************************************\n\n");

  if (iargc < 3 || iargc > 9) {

    printf("  convert2syrthes [-v 4] [-dim dimension]  -m mesh1.ext [-o mesh2.syr]  \n");
    printf("                  with  .ext = \n");
    printf("                               .neu (GAMBIT format)\n");
    printf("                               .msh (GMSH format)\n");
    printf("                               .med (MED format)\n");
    printf("     --> changes initial mesh file format to SYRTHES 3.4 mesh format\n");
    printf("\n  Additional options :\n");
    printf("     -o mesh2.syr : to change the name of the file\n");
    printf("     -dim 2 or -dim 3 : needed for GMSH file format, ignored in the other cases\n");
    printf("     -v 4 : result will be a SYRTHES 4 mesh file format\n");
    printf("\n  Note :\n");
    printf("     For med format, convert2syrthes creates an additionnal ASCII file (..._desc)\n");
    printf("     where you will find the correspondances between group names (used in Salome)\n");
    printf("     and familly (or references) numbers used in SYRTHES\n");
    printf("\n  Examples :\n");
    printf("     convert2syrthes -m square.med                         --> square.syr square.syr_desc\n");
    printf("     convert2syrthes -m square.neu                         --> square.syr\n");
    printf("     convert2syrthes -m square.msh -dim 2                  --> square.syr\n");
    printf("     convert2syrthes -m square.med -o square2.syr          --> square2.syr square2.syr_desc\n");
    
/*    printf("     - pour ext=neu ou msh, si '-ref' est absent, les couleurs des noeuds sont ignorees\n"); */
/*    printf("                            si '-ref' est present, les couleurs des noeuds sont prises en compte\n");     */
    return 1;

  }
 
  /* initialisations des entiers */
  extension = 0;
  maillage.dimension = 0; /* dimension de l'espace  2 ou 3 */
  maillage.dim_elem = 0; /* dimension des elements 1, 2 ou 3 */
  maillage.nbnoeuds = 0; /* nombre de noeuds */
  maillage.nbelem = 0; /* nombre d'elements */
  maillage.nbnoeuds_par_elem = 0; /* nombre de noeuds par element */
  maillage.nbelem_de_bord = 0; /* nombre d'elements de bord pour version Syrthes 4.0 */
  maillage.numligne = 0; /* numero de ligne lue dans le maillage de lecture */
  maillage.nbzones = 0; /* nombre de zones */
  maillage.nbcl = 0; /* nombre de conditions limites */
  maillage.fichier_ext = NULL;
  maillage.fichier_syr = NULL;
  maillage.fichier_desc = NULL;
  

  /* gestion de l'extension */
  m=1;
  while (strcmp(iargv[m],"-m")!=0 && m<iargc-1) m++;
  if (m==iargc-1) {
    printf("ERROR : wrong list of parameters\n");
    printf("  convert2syrthes [-v 4] -m mesh1.ext [-o mesh2.syr] [-dim dimension] \n");
    return 1;
  }
  extension = 0;
  longueur = strlen(iargv[m+1]);
  if (longueur > 3) {
    if ( (iargv[m+1][longueur-4]=='.') && (iargv[m+1][longueur-3]=='n') &&
         (iargv[m+1][longueur-2]=='e') && (iargv[m+1][longueur-1]=='u')) extension = 1; /* Gambit neutral */
    else if ( (iargv[m+1][longueur-4]=='.') && (iargv[m+1][longueur-3]=='m') &&
              (iargv[m+1][longueur-2]=='s') && (iargv[m+1][longueur-1]=='h')) extension = 2; /* GMSH */
    else if ( (iargv[m+1][longueur-4]=='.') && (iargv[m+1][longueur-3]=='m') &&
              (iargv[m+1][longueur-2]=='e') && (iargv[m+1][longueur-1]=='d')) extension = 3; /* MED */
  }

  /* version Syrthes a ecrire */
  maillage.version_syr = 3; /* 3.4 par defaut */
  v=1;
  while (strcmp(iargv[v],"-v")!=0 && v<iargc-1) v++;
  if (v<iargc-1) {
    if (iargv[v+1][0] == '4') {
      maillage.version_syr = 4;
      printf("Convert to SYRTHES 4 format\n");
    }
    else
      printf("Convert to SYRTHES 3.4 format\n");
  }
  else
    printf("Convert to SYRTHES 3.4 format\n");
 
  /* gestion de l'option -ref : option annulee*/
  ref=1;
/*  while (strcmp(iargv[ref],"-ref")!=0 && ref<iargc-1) ref++;
  if (strcmp(iargv[ref],"-ref")==0 && extension==1) {
    printf("option -ref detectee, prise en compte des couleurs de noeuds (a partir de Gambit Neutral) :");
    printf(" le fichier Syrthes sera modifie apres une premiere conversion. \n");
  }
  else if (strcmp(iargv[ref],"-ref")==0 && extension==2) {
    printf("option -ref detectee, prise en compte des couleurs de noeuds (a partir de GMSH)\n");
    printf(" le fichier Syrthes sera modifie apres une premiere conversion. \n");
  }
  else if (strcmp(iargv[ref],"-ref")==0)
    printf("Option -ref ignoree car le format de lecture n'est ni Gambit Neutral ni GMSH\n");
*/
    
  if (extension == 2) {
    /* si GMSH lecture de la dimension "-dim 2 ou -dim 3" */
    dim=1;
    while (strcmp(iargv[dim],"-dim")!=0 && dim<iargc-1) dim++;
    if (dim<iargc-1) {
      if (iargv[dim+1][0] == '3') maillage.dimension = 3;
      else if (iargv[dim+1][0] == '2') maillage.dimension = 2;
      else {
        printf("ERROR : wrong value for the space dimension : %c\n",iargv[dim+1][0]);
        return 1;
      }	
    }
    else {
      printf("ERROR : to convert GMSH file format, you have to indicate the space dimension in arguments (-dim 2 ou -dim 3) \n");
      printf("  convert2syrthes [-v 4]-dim dimension  -m mesh1.ext [-o mesh2.syr]\n\n");
    return 1;
    }
  }

  if (extension !=0) {
    /* gestion de l'option -o */
    o=1;
    while (strcmp(iargv[o],"-o")!=0 && o<iargc-1) o++;
    if (o<iargc-1) {
      nom_syr = strdup(iargv[o+1]);
    }
    else {
      /* nom de fichier de sortir base sur le nom du fichier d'entree */
      buffer = strdup(iargv[m+1]);
      buffer = basename(buffer);
      nom_syr = strcat(strtok(buffer,"."),".syr");
    }
    /* ouverture du fichier Syrthes en ecriture */
    if (ouvrir_syr(&maillage,nom_syr) != 0) return 1;
  }
  
  /* conversion selon extension */
  switch (extension) {
  case 1 : if (conversion_neu(&maillage,iargv[m+1]) != 0) return 1; break;
  case 2 : if (conversion_msh(&maillage,iargv[m+1]) != 0) return 1; break;
#ifdef MED
  case 3 : if (conversion_med(&maillage,iargv[m+1], nom_syr) != 0) return 1; break;
  default : printf("ERROR : wrong type of mesh format\n");
            printf("        check the file name extension (must be .neu  .msh or .med)\n") ; 
	    return 1;
#else
  default : printf("ERROR : wrong type of mesh format\n");
            printf("        check the file name extension (must be .neu  or .msh)\n") ; 
	    return 1;
#endif
  }
  
  /* fermeture du fichier Syrthes */
  if (fermer_syr(&maillage) != 0) return 1;

  /* reouverture du fichier Syrthes pour au moins l'entete*/
  if (reouvrir_syr(&maillage, nom_syr) != 0) return 1;
  
  /* reecriture de l'entete avec les bonnes informations cette fois-ci */
  printf("  Modification of the head file...\n");
  if (ecrire_entete_syr(&maillage) != 0) return 1;
   
  /* reecriture des noeuds avec la couleur */
/*  if (strcmp(iargv[ref],"-ref")==0 && extension==1) {*/
  if (extension==1) {
    printf("  Modification of the SYRTHES for the nodes references...\n");
    if (ouvrir_neu(&maillage,iargv[m+1]) != 0) return 1;
    if (modif_noeuds_neusyr(&maillage) != 0) return 1;
    if (fermer_neu(&maillage) != 0) return 1;
  }
/*  else if (strcmp(iargv[ref],"-ref")==0 && extension==2) {*/
  else if (extension==2) {
    printf("  Modification of the SYRTHES for the nodes references...\n");
    if (ouvrir_msh(&maillage,iargv[m+1]) != 0) return 1;
    if (modif_noeuds_mshsyr(&maillage) != 0) return 1;
    if (fermer_msh(&maillage) != 0) return 1;
  }
  
  /* fermeture du fichier Syrthes */
  if (fermer_syr(&maillage) != 0) return 1;
  
  printf("\nSYRTHES file conversion OK.\n\n");
  return 0;
  /* sortie normale */

}
