/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.tomcat.util.http;

import java.io.Serial;

import jakarta.servlet.http.HttpServletResponse;

/**
 * Extend {@link IllegalStateException} to identify the cause as an invalid parameter.
 * <p>
 * Implementation note: This class extends {@link IllegalStateException} since that is the class that the Servlet 6.1
 * onwards Javadocs define is thrown by the various {@code ServletRequest.getParameterXXX()} methods.
 */
public class InvalidParameterException extends IllegalStateException {

    @Serial
    private static final long serialVersionUID = 1L;

    private final int errorCode;


    /**
     * Construct a new exception with the given message.
     *
     * @param message The message to use for the exception
     */
    public InvalidParameterException(String message) {
        this(message, HttpServletResponse.SC_BAD_REQUEST);
    }


    /**
     * Construct a new exception with the given message and error code.
     *
     * @param message   The message to use for the exception
     * @param errorCode The HTTP status code to use when reporting this error. Expected to be >= 400.
     */
    public InvalidParameterException(String message, int errorCode) {
        this(message, null, errorCode);
    }


    /**
     * Construct a new exception with the given message and cause.
     *
     * @param message The message to use for the exception
     * @param cause   The exception to use as the cause of this exception
     */
    public InvalidParameterException(String message, Throwable cause) {
        this(message, cause, HttpServletResponse.SC_BAD_REQUEST);
    }


    /**
     * Construct a new exception with the given cause. The message for this exception will be generated by calling
     * {@code cause.toString()}.
     *
     * @param cause The exception to use as the cause of this exception
     */
    public InvalidParameterException(Throwable cause) {
        this(cause.toString(), cause, HttpServletResponse.SC_BAD_REQUEST);
    }


    /**
     * Construct a new exception with the given cause and error code. The message for this exception will be generated
     * by calling {@code cause.toString()}.
     *
     * @param cause     The exception to use as the cause of this exception
     * @param errorCode The HTTP status code to use when reporting this error. Expected to be >= 400.
     */
    public InvalidParameterException(Throwable cause, int errorCode) {
        this(cause.toString(), cause, errorCode);
    }


    private InvalidParameterException(String message, Throwable cause, int errorCode) {
        super(message, cause);
        this.errorCode = errorCode;
    }


    public int getErrorCode() {
        return errorCode;
    }
}